<?php

namespace App\Services;


use App\Entity\MailNotification;
use App\Entity\User;
use App\Repository\ScheduleRepository;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\Entity;
use Doctrine\ORM\NonUniqueResultException;
use Exception;
use ReflectionException;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\WebpackEncoreBundle\Asset\EntrypointLookupInterface;
use Twig\Environment;
use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;

class Mailer
{

    private MailerInterface $mailer;
    private EntrypointLookupInterface $entrypointLookup;
    private Environment $twig;
    private string $app_mail_address;
    private MpdfService $mpdfService;
    private string $publicDir;
    private EntityManagerInterface $entityManager;
    private EveningCheckService $eveningCheckService;
    private ScheduleRepository $scheduleRepository;
    private string $app_mail_ending;
    private UrlGeneratorInterface $urlGenerator;

    public function __construct(string $app_mail_address, string $publicDir, string $app_mail_ending,  MailerInterface $mailer, EntityManagerInterface $entityManager,
                                EveningCheckService $eveningCheckService, ScheduleRepository $scheduleRepository,
                                EntrypointLookupInterface $entrypointLookup, Environment $twig, MpdfService $mpdfService, UrlGeneratorInterface $urlGenerator)
    {

        $this->mailer = $mailer;
        $this->entrypointLookup = $entrypointLookup;
        $this->twig = $twig;
        $this->app_mail_address = $app_mail_address;
        $this->mpdfService = $mpdfService;
        $this->publicDir = $publicDir;
        $this->entityManager = $entityManager;
        $this->eveningCheckService = $eveningCheckService;
        $this->scheduleRepository = $scheduleRepository;
        $this->app_mail_ending = $app_mail_ending;
        $this->urlGenerator = $urlGenerator;
    }

    public function saveMailNotification(User $user, string $subject, DateTimeImmutable $date, int $type, array $options = []): void
    {
        $mailNotification = new MailNotification();
        $mailNotification->setSent(false);
        $mailNotification->setType($type);
        $mailNotification->setReceiver($user);
        $mailNotification->setSubject($subject);
        $mailNotification->setCreationTime($date);
        $mailNotification->setOptions($options);
        $this->entityManager->persist($mailNotification);
    }

    /**
     * @throws SyntaxError
     * @throws TransportExceptionInterface
     * @throws ReflectionException
     * @throws RuntimeError
     * @throws NonUniqueResultException
     * @throws LoaderError
     */
    public function sendMailNotification(MailNotification $mailNotification): void
    {
        switch ($mailNotification->getType()) {
            case MailNotification::NEW_STUDENT_NOTE_TYPE: {
                $this->sendNewStudentNoteNoticeToEducator($mailNotification);
                break;
            }
            case MailNotification::COMPLETE_EVENING_CHECK_TYPE: {
                $this->sendCompleteDailyReport($mailNotification);
                break;
            }
            case MailNotification::INCOMPLETE_EVENING_CHECK_TYPE: {
                $this->sendIncompleteDailyReport($mailNotification);
                break;
            }
            case MailNotification::NEW_USER_TYPE: {
                $this->sendNewUserRegistration($mailNotification);
                break;
            }
            default: {
                return;
            }
        }
    }
    /**
     * @throws SyntaxError
     * @throws TransportExceptionInterface
     * @throws ReflectionException
     * @throws RuntimeError
     * @throws NonUniqueResultException
     * @throws LoaderError
     */
    private function sendCompleteDailyReport(MailNotification $mailNotification): void
    {
        $this->sendDailyReport($mailNotification, MailNotification::COMPLETE_EVENING_CHECK_TYPE);
    }

    /**
     * @throws SyntaxError
     * @throws TransportExceptionInterface
     * @throws ReflectionException
     * @throws RuntimeError
     * @throws NonUniqueResultException
     * @throws LoaderError
     */
    private function sendIncompleteDailyReport(MailNotification $mailNotification): void
    {
        $this->sendDailyReport($mailNotification, MailNotification::INCOMPLETE_EVENING_CHECK_TYPE);
    }

    /**
     * @throws SyntaxError
     * @throws TransportExceptionInterface
     * @throws RuntimeError
     * @throws LoaderError
     * @throws ReflectionException
     * @throws NonUniqueResultException
     * @throws Exception
     */
    private function sendDailyReport(MailNotification $mailNotification, int $type): void
    {
        $schedules = $this->scheduleRepository->findScheduleForDate($mailNotification->getCreationTime());
        $countAllActiveStudents = $this->eveningCheckService->countAllActiveStudents();
        $absentStudents = $this->eveningCheckService->getAbsentStudentsForDate($mailNotification->getCreationTime());

        $this->entrypointLookup->reset();
        $html = $this->twig->render('email/daily_attendance_record.html.twig', [
            'mailEnding' => $this->app_mail_ending,
            'schedules' => $schedules,
            'countAllStudents' => $countAllActiveStudents,
            'absentStudents' => $absentStudents,
            'publicDir' => $this->publicDir,
            'studentsWithoutEveningCheck' => $type === MailNotification::COMPLETE_EVENING_CHECK_TYPE ? null : $this->eveningCheckService->getStudentsWithoutEveningCheckForDate($mailNotification->getCreationTime()),
        ]);
        $mpdf = $this->mpdfService->setMpdf();
        $mpdf->WriteHTML($html);
        $content = $mpdf->Output('', 'S');
        $email = (new TemplatedEmail())
            ->from($this->app_mail_address)
            ->to($mailNotification->getReceiver()->getEmail())
            ->subject($mailNotification->getSubject())
            ->htmlTemplate('email/evening_check.html.twig')
            ->context([
                'mailEnding' => $this->app_mail_ending,
            ])
            ->attach($content, sprintf('daily-attendance-report-%s.pdf', (new DateTimeImmutable())->format('Y-m-d')));;

        $this->mailer->send($email);
        $mailNotification->setSent(true);
        $this->entityManager->flush();
    }

    /**
     * @throws TransportExceptionInterface
     */
    private function sendNewStudentNoteNoticeToEducator(MailNotification $mailNotification): void
    {
        if ($mailNotification->getReceiver() === null) {
            return;
        }
        $email = (new TemplatedEmail())
            ->from($this->app_mail_address)
            ->to($mailNotification->getReceiver()->getEmail())
            ->subject($mailNotification->getSubject())
            ->htmlTemplate('email/new_student_note_notice.html.twig')
            ->context([
                'options' => $mailNotification->getOptions(),
                'mailEnding' => $this->app_mail_ending,
            ]);

        $this->mailer->send($email);
        $mailNotification->setSent(true);
        $this->entityManager->flush();
    }

    /**
     * @throws TransportExceptionInterface
     */
    private function sendNewUserRegistration(MailNotification $mailNotification): void
    {
        $email = (new TemplatedEmail())
            ->from($this->app_mail_address)
            ->to($mailNotification->getReceiver()->getEmail())
            ->subject($mailNotification->getSubject())
            ->htmlTemplate('email/new_user.html.twig')
            ->context([
                'siteAddress' => $this->urlGenerator->generate('app_login', [], UrlGeneratorInterface::ABSOLUTE_URL),
                'mailEnding' => $this->app_mail_ending,
            ]);

        $this->mailer->send($email);
        $mailNotification->setSent(true);
        $this->entityManager->flush();
    }
}