<?php
namespace App\Services;

use App\Controller\Educator\Crud\MeetingCrudEducatorController;
use App\Controller\Educator\Crud\StudentNoteCrudEducatorController;
use App\Controller\Student\Crud\MeetingCrudStudentController;
use App\Controller\Student\StudentDashboardController;
use App\Entity\MailNotification;
use App\Entity\Meeting;
use App\Entity\MessageReceiver;
use App\Entity\Student;
use App\Entity\StudentNote;
use BenTools\WebPushBundle\Model\Message\PushNotification;
use DateTimeImmutable;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Contracts\Translation\TranslatorInterface;

class NotificationService {

    private UserSubscriptionManager $userSubscriptionManager;
    private AdminUrlGenerator $adminUrlGenerator;
    private PushNotifier $pushNotifier;
    private TranslatorInterface $translator;

    public function __construct(UserSubscriptionManager $userSubscriptionManager, AdminUrlGenerator $adminUrlGenerator,
                                PushNotifier $pushNotifier, TranslatorInterface $translator)
    {
        $this->userSubscriptionManager = $userSubscriptionManager;
        $this->adminUrlGenerator = $adminUrlGenerator;
        $this->pushNotifier = $pushNotifier;
        $this->translator = $translator;
    }
    /**
     * @throws Exception
     */
    public function createNotificationsForStudentNote(Mailer $mailer, StudentNote $studentNote): void
    {
        $options = [
            'student' => $studentNote->getStudent()->__toString(),
            'educator' => $studentNote->getEducator()->__toString(),
            'note' => $studentNote->getNote(),
        ];
        $mailer->saveMailNotification($studentNote->getStudent()->getEducator()->getUser(), $this->translator->trans('entity.mail_notification.new_student_note'), new DateTimeImmutable(), MailNotification::NEW_STUDENT_NOTE_TYPE, $options);

        $educator = $studentNote->getStudent()->getEducator();
        $subscriptions = $this->userSubscriptionManager->findByUser($educator->getUser());
        $url = $this->adminUrlGenerator->setController(StudentNoteCrudEducatorController::class)
            ->setDashboard($educator->getUser()->getDashboard())
            ->setAction('detail')
            ->setEntityId($studentNote->getId())
            ->generateUrl();
        $notification = new PushNotification($this->translator->trans('entity.push_notification.new_student_note'), [
            PushNotification::BODY => $this->translator->trans('entity.push_notification.new_student_note_text', [
                '%student%' => $studentNote->getStudent(),
            ]),
            PushNotification::REQUIREINTERACTION => true,
            PushNotification::SILENT => false,
            PushNotification::SOUND => true,
            PushNotification::DATA => ['link' => $url],
        ]);
        $this->pushNotifier->saveNewNotification($notification, $subscriptions);
    }

    /**
     * @throws Exception
     */
    public function createNotificationForUnsentMessages(MessageReceiver $messageReceiver): void
    {
        $user = $messageReceiver->getReceiver();
        $subscriptions = $this->userSubscriptionManager->findByUser($user);
        $message = $messageReceiver->getMessage();
        $url = $this->adminUrlGenerator->setDashboard($user->getDashboard())->setRoute('app_message_detail')->set('id', $message->getId())->generateUrl();
        $notification = new PushNotification($this->translator->trans('entity.message.create'), [
            PushNotification::BODY => $this->translator->trans('entity.push_notification.received_new_message'),
            PushNotification::REQUIREINTERACTION => true,
            PushNotification::SILENT => false,
            PushNotification::SOUND => true,
            PushNotification::DATA => ['link' => $url],
        ]);
        $this->pushNotifier->saveNewNotification($notification, $subscriptions);
    }

    /**
     * @throws Exception
     */
    public function createNotificationForMeeting(Meeting $meeting, bool $isEducator): void
    {
        $subscriptions = [];
        if ($isEducator) {
            $educator = $meeting->getEducator();
            $subscriptions = $this->userSubscriptionManager->findByUser($educator->getUser());
        }
        else {
            foreach ($meeting->getMeetingAttendances() as $meetingAttendance) {
                $studentSubscriptions = $this->userSubscriptionManager->findByUser($meetingAttendance->getStudent()->getUser());
                $subscriptions = array_merge($subscriptions, $studentSubscriptions);
            }
        }
        $type = count($meeting->getMeetingAttendances()) == 1? $this->translator->trans('entity.push_notification.individual') : $this->translator->trans('entity.push_notification.group');
        $url = $this->adminUrlGenerator->setDashboard($isEducator? $educator->getUser()->getDashboard() : StudentDashboardController::class)
            ->setController($isEducator? MeetingCrudEducatorController::class : MeetingCrudStudentController::class)
            ->setAction(Action::DETAIL)
            ->setEntityId($meeting->getId())
            ->generateUrl();
        $notification = new PushNotification($this->translator->trans('entity.push_notification.today_meeting'), [
            PushNotification::BODY => $this->translator->trans('entity.push_notification.today_meeting_text', [
                '%type%' => $type,
                '%time%' => $meeting->getPlannedDate()->format('H:i'),
            ]),
            PushNotification::REQUIREINTERACTION => true,
            PushNotification::SILENT => false,
            PushNotification::SOUND => true,
            PushNotification::DATA => ['link' => $url],
        ]);
        $this->pushNotifier->saveNewNotification($notification, $subscriptions);
    }
    public function createNotificationStudentNotReturned(?Student $student, string $action): void
    {
        $educator = $student->getEducator();
        $subscriptions = $this->userSubscriptionManager->findByUser($educator->getUser());
        $translatedAction = $this->translator->trans($action);
        $notification = new PushNotification($this->translator->trans('entity.push_notification.not_returned_student'), [
            PushNotification::BODY => $this->translator->trans('entity.push_notification.not_returned_student_text', [
                '%action%' => $translatedAction,
                '%student%' => $student,
            ]),
            PushNotification::REQUIREINTERACTION => true,
            PushNotification::SILENT => false,
            PushNotification::SOUND => true,
        ]);
        $this->pushNotifier->saveNewNotification($notification, $subscriptions);
    }
}
