<?php
namespace App\Services;

use App\Entity\Educator;
use App\Entity\Schedule;
use App\Repository\ScheduleRepository;
use DateInterval;
use DatePeriod;
use DateTime;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Exception;

class ScheduleService {


    /**
     * @throws Exception
     */
    public static function findSchoolWeek(array $week): int
    {
        $startDate = new DateTime();
        $startDate->setISODate($week['year'], $week['week']);

        $endDate = clone $startDate;
        $endDate->modify('+6 days');

        $septemberFirst = new DateTime("{$week['year']}-09-01");

        if (($septemberFirst >= $startDate && $septemberFirst <= $endDate) || $startDate->format('m') >= 9) {
            return $septemberFirst->format('W');
        }

        $previousYear = $week['year'] - 1;
        $septemberFirst = new DateTime($previousYear. "-09-01");

        return $septemberFirst->format('W');
    }

    /**
     * @throws Exception
     */
    public static function getWeekOfCycle(array $week, int $schoolWeek, int $maxCycle): int
    {
        $date = new DateTime();
        $date->setISODate($week['year'], $week['week']);
        $startDate = new DateTime();
        $startDate->setISODate($week['year'], $week['week']);

        $endDate = clone $startDate;
        $endDate->modify('+6 days');

        $septemberFirst = new DateTime("{$week['year']}-09-01");

        if ((($septemberFirst >= $startDate && $septemberFirst <= $endDate) || $startDate->format('m') >= 9)) {
            return ($week['week'] - $schoolWeek + 1) % $maxCycle;
        }

        //year before
        $lastDayOfYear = new DateTime($week['year']-1 ."-12-31");
        $lastWeekOfCycle = ($lastDayOfYear->format('W') - $schoolWeek + 1) % $maxCycle;
        $firstWeekOfCycle = $lastWeekOfCycle + 1;
        return ($week['week'] - 1 + $firstWeekOfCycle) % $maxCycle;
    }
    public static function isValidTimeCombination($fromTime, $toTime): bool
    {
        foreach (Schedule::TRANSFORM_SCHEDULE as $scheduleKey => $scheduleData) {
            if ($scheduleData['from'] === $fromTime && $scheduleData['to'] === $toTime) {
                return true;
            }
        }
        return false;
    }

    /**
     * @throws Exception
     */
    public function checkIfFreeDay(DateTimeImmutable $date): bool {
        $year = (int) $date->format('Y');

        $date_formatted = $date->format('Y-m-d');
        foreach (ScheduleService::get_holidays($year) as $holidayPeriod) {
            foreach ($holidayPeriod as $holiday) {
                $holiday_formatted = $holiday->format('Y-m-d');
                if ($date_formatted === $holiday_formatted) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @throws Exception
     */
    private static function get_holidays(int $year): array {
        $holidays = [];

        // Easter
        $easter = new DateTimeImmutable("$year-03-21");
        $easterMonday = $easter->modify('+' . easter_days($year)+1 . ' days');
        $easterThursday = $easterMonday->modify('-4 days');
        $holidays[] = new DatePeriod($easterThursday, new DateInterval('P1D'), $easterMonday);

        // Christmas
        $startChristmas = new DateTimeImmutable("$year-12-23");
        if ($startChristmas->format('N') == 1 || $startChristmas->format('N') >= 6) {
            $startChristmas = $startChristmas->modify('previous friday');
        }
        $holidays[] = new DatePeriod($startChristmas, new DateInterval('P1D'), new DateTimeImmutable("$year-12-31"));

        $endChristmas = new DateTimeImmutable("$year-01-03");
        if ($endChristmas->format('N') >= 6) {
            $endChristmas = $endChristmas->modify('next monday');
        }
        $holidays[] = new DatePeriod(new DateTimeImmutable("$year-01-01"), new DateInterval('P1D'), $endChristmas);

        // Summer holidays
        $startSummer = new DateTimeImmutable("$year-07-01");
        $endSummer = new DateTimeImmutable("$year-09-01");

        if ($startSummer->format('N') >= 6) {
            $startSummer = $startSummer->modify('next monday');
        }
        if ($endSummer->format('N') >= 6) {
            $endSummer = $endSummer->modify('previous friday');
        }
        $holidays[] = new DatePeriod($startSummer, new DateInterval('P1D'), $endSummer);
        return $holidays;
    }

}
